import tkinter as tk
from tkinter import filedialog, messagebox
import os


class AlphaCodeStudio:
    def __init__(self, root):
        self.root = root
        self.root.title("Alpha Code Studio")
        self.root.geometry("800x600")
        self.filename = None

        # Create Text Widget
        self.text_area = tk.Text(root, undo=True, bg="#1e1e1e", fg="white", insertbackground="white", font=("Consolas", 12))
        self.text_area.pack(fill=tk.BOTH, expand=1)

        # Scrollbar
        self.scroll = tk.Scrollbar(self.text_area)
        self.scroll.pack(side=tk.RIGHT, fill=tk.Y)
        self.scroll.config(command=self.text_area.yview)
        self.text_area.config(yscrollcommand=self.scroll.set)

        # Menu
        self.menu_bar = tk.Menu(root)
        self._create_file_menu()
        self.root.config(menu=self.menu_bar)

        # Basic Syntax Highlight
        self.text_area.bind("<KeyRelease>", self.syntax_highlight)

    def _create_file_menu(self):
        file_menu = tk.Menu(self.menu_bar, tearoff=False)
        file_menu.add_command(label="New", command=self.new_file)
        file_menu.add_command(label="Open", command=self.open_file)
        file_menu.add_command(label="Save", command=self.save_file)
        file_menu.add_command(label="Save As", command=self.save_as)
        file_menu.add_separator()
        file_menu.add_command(label="Exit", command=self.root.quit)
        self.menu_bar.add_cascade(label="File", menu=file_menu)

    def new_file(self):
        self.filename = None
        self.text_area.delete(1.0, tk.END)
        self.root.title("Alpha Code Studio - New File")

    def open_file(self):
        self.filename = filedialog.askopenfilename(defaultextension=".py",
                                                   filetypes=[("Python Files", "*.py"), ("All Files", "*.*")])
        if self.filename:
            with open(self.filename, "r", encoding="utf-8") as file:
                self.text_area.delete(1.0, tk.END)
                self.text_area.insert(1.0, file.read())
            self.root.title(f"Alpha Code Studio - {os.path.basename(self.filename)}")
            self.syntax_highlight()

    def save_file(self):
        if self.filename:
            with open(self.filename, "w", encoding="utf-8") as file:
                file.write(self.text_area.get(1.0, tk.END))
        else:
            self.save_as()

    def save_as(self):
        self.filename = filedialog.asksaveasfilename(defaultextension=".py",
                                                     filetypes=[("Python Files", "*.py"), ("All Files", "*.*")])
        if self.filename:
            self.save_file()
            self.root.title(f"Alpha Code Studio - {os.path.basename(self.filename)}")

    def syntax_highlight(self, event=None):
        # Basic Python keyword highlighting
        keywords = ["def", "class", "import", "from", "return", "if", "else", "elif",
                    "while", "for", "try", "except", "with", "as", "pass", "break", "continue", "True", "False"]
        self.text_area.tag_remove("keyword", "1.0", tk.END)
        for keyword in keywords:
            idx = "1.0"
            while True:
                idx = self.text_area.search(rf"\b{keyword}\b", idx, nocase=False, stopindex=tk.END, regexp=True)
                if not idx:
                    break
                lastidx = f"{idx}+{len(keyword)}c"
                self.text_area.tag_add("keyword", idx, lastidx)
                idx = lastidx
        self.text_area.tag_config("keyword", foreground="#569CD6")

# Run app
if __name__ == "__main__":
    root = tk.Tk()
    app = AlphaCodeStudio(root)
    root.mainloop()

